/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

        $Id: CLDAPPGPKeyServer.cpp,v 1.12 2001/03/24 14:20:24 jason Exp $
____________________________________________________________________________*/
#include <string.h>

#include "pgpConfig.h"
#include "CLDAPPGPKeyServer.h"
#include "pgpEventPriv.h"
#include "pgpKeys.h"
#include "pgpUtilities.h"
#include "pgpMem.h"
#include "pgpStrings.h"


#define ThrowIfLDAPCanceled_()	if (mCanceled) \
									ThrowPGPError_(kPGPError_UserAbort);
#define ThrowIfPGPErrorOrLDAPCanceled_(x)	{	ThrowIfPGPError_(x); \
												ThrowIfLDAPCanceled_(); }

#define kNamingContextsAttr			"namingContexts"
#define kEmptyBaseDN				""
#define kObjectclassAny				"(objectclass=*)"
#define kPGPServerInfoCN			"cn=PGPServerInfo"
#define kBaseKeySpaceAttr			"pgpBaseKeySpaceDN"
#define kPGPKeyObjectClass			"pgpkey"
#define kObjectclassAttr			"objectclass"
#define kPGPKeyAttr					"pgpkey"
#define kPGPCertIDAttr				"pgpcertid"
#define kPGPDisabledAttr			"pgpdisabled"
#define kPGPKeyIDAttr				"pgpkeyid"
#define kPGPKeyTypeAttr				"pgpkeytype"
#define kPGPUserIDAttr				"pgpuserid"
#define kPGPKeyCreateTime			"pgpkeycreatetime"
#define kPGPKeyExpireTime			"pgpkeyexpiretime"
#define kPGPSignerIDAttr			"pgpsignerid"
#define kPGPRevokedAttr				"pgprevoked"
#define kPGPSubKeyIDAttr			"pgpsubkeyid"
#define kPGPKeySizeAttr				"pgpkeysize"
#define kPGPNumberOfAttrs			13

#define kPGPAlgorithmString_RSA		"RSA"
#define kPGPAlgorithmString_DSSDH	"DSS/DH"

#define kMaxKeySizeStringLength		6	/* 5 for the number, 1 for the '\0' */

CLDAPPGPKeyServer::CLDAPPGPKeyServer(
	PGPContextRef				inContext,
	const char *				inHostName,
	PGPUInt32					inHostAddress,
	PGPUInt16					inHostPort,
	PGPKeyServerProtocol		inType,
	PGPKeyServerClass			inClass,
	PGPKeyServerAccessType		inAccessType )
	: CKeyServer( inContext, inHostName, inHostAddress, inHostPort, NULL, inType, inClass ),
	  mContext( inContext ),
	  mAccessType( inAccessType ), 
	  mLDAP( kInvalidPGPldapContextRef ),
	  mBaseKeySpaceDN( NULL ),
	  mExportFormat( kPGPExportFormat_Complete )
{
	return;
}


CLDAPPGPKeyServer::~CLDAPPGPKeyServer( )
{
	return;
}


	void
CLDAPPGPKeyServer::Cancel( )
{
	CKeyServer::Cancel();

	if( !PGPldapContextRefIsValid( mLDAP ) )
		return;

	(void) PGPldapUnbind( mLDAP );
	
	(void) PGPFreeLDAPContext( mLDAP );
	mLDAP = kInvalidPGPldapContextRef;

	return;
}


	void
CLDAPPGPKeyServer::Open(
	PGPtlsSessionRef	inTLSSession )
{
	StPreserveSocketsEventHandler		preserve(this);
	PGPldapResult						result;
	char *								szError			= NULL;
	const char *						namingContextsAttrs[] = { kNamingContextsAttr,
															NULL };
	char *								serverInfoCN	= NULL;
	PGPldapMessageRef					firstMessage	= kInvalidPGPldapMessageRef;
	PGPldapMessageRef					currentMessage	= kInvalidPGPldapMessageRef;
	char **								namingContexts	= NULL;
	char **								keySpaceAttrs	= NULL;
	PGPUInt32							namingContextIndex	= 0;
	char	**							values			= NULL;
	PGPUInt32							i				= 0;
	PGPSocketRef						sock			= kInvalidPGPSocketRef;
	PGPError							err				= kPGPError_NoErr;

	SetErrorString( NULL );
	try
	{
		CKeyServer::Open(inTLSSession);

		err = pgpEventKeyServer( mContext,
					mEventHandler,
					mEventHandlerData,
					(PGPKeyServerRef) this,
					kPGPKeyServerState_Opening );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewLDAPContext( mContext, &mLDAP );
		if( !PGPldapContextRefIsValid( mLDAP ) || IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		err = PGPldapOpen( mLDAP, mHostName, mHostPort );
		ThrowIfLDAPCanceled_();

		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		// Secure connection for LDAPS
		if( mType == kPGPKeyServerProtocol_LDAPS)
		{
			err = PGPldapGetOption( mLDAP, kPGPldapOpt_Desc, &sock );
			if( IsPGPError( err ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			if( ( mTLSSession == kInvalidPGPtlsSessionRef ) || 
				( PGPSocketsEstablishTLSSession(
					(PGPSocketRef) sock, mTLSSession ) 
				!= kPGPError_NoErr ) )
			{
				err = pgpEventKeyServerTLS(
								mContext,
								mEventHandler,
								mEventHandlerData,
								(PGPKeyServerRef) this,
								kPGPKeyServerState_TLSUnableToSecureConnection,
								mTLSSession );
			}
			else
			{
				err = pgpEventKeyServerTLS(
								mContext,
								mEventHandler,
								mEventHandlerData,
								(PGPKeyServerRef) this,
								kPGPKeyServerState_TLSConnectionSecured,
								mTLSSession );

				mSecured = true;
			}
			ThrowIfPGPErrorOrLDAPCanceled_( err );
		}

		mIsOpen = true;
	
		/* Look for the keyspace DN */

		err = PGPNewLDAPMessage( mLDAP, &firstMessage );
		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );
		
		err = PGPldapSearchSync( mLDAP,
				(char *) kEmptyBaseDN,
				kPGPldapScope_Base,
				(char *) kObjectclassAny,
				(char **) namingContextsAttrs,
				FALSE,
				firstMessage );
		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		err = PGPldapGetValues( mLDAP,
				firstMessage,
				(char *) kNamingContextsAttr,
				&namingContexts );
		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		if( PGPldapMessageRefIsValid( firstMessage ) )
		{
			(void) PGPFreeLDAPMessage( firstMessage );
			firstMessage = kInvalidPGPldapMessageRef;
		}

		for( namingContextIndex = 0; 
			IsntNull( namingContexts[namingContextIndex] ) &&
			IsNull( mBaseKeySpaceDN );
			namingContextIndex++ )
		{
			err = PGPNewLDAPMessage( mLDAP, &firstMessage );
			if( IsPGPError( err ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			serverInfoCN = (char *) PGPNewData( 
					PGPPeekContextMemoryMgr( mContext ),
					strlen( namingContexts[namingContextIndex] ) +
						strlen( ", " ) +
						strlen( kPGPServerInfoCN ) + 1 /* '\0' */,
					kPGPMemoryMgrFlags_Clear );
			if( IsNull( serverInfoCN ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			sprintf( serverInfoCN, "%s, %s", kPGPServerInfoCN, 
					namingContexts[namingContextIndex] );
			err = PGPldapSearchSync( mLDAP,
					serverInfoCN,
					kPGPldapScope_Base,
					(char *) kObjectclassAny,
					keySpaceAttrs,
					FALSE,
					firstMessage );
			switch( err )
			{
				case kPGPError_NoErr:
					break;
				
				case kPGPError_LDAPInsufficientAccess:
					ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
					break;
				
				default:
					ThrowPGPError_( kPGPError_ServerOpenFailed );
					break;
			}

			if( IsntNull( serverInfoCN ) )
				(void) PGPFreeData( serverInfoCN );

			err = PGPldapFirstEntry( mLDAP, firstMessage, &currentMessage );
			if( IsPGPError( err ) )
				ThrowPGPError_( kPGPError_ServerOpenFailed );

			while( PGPldapMessageRefIsValid( currentMessage ) && IsNull( mBaseKeySpaceDN ) )
			{
				err = PGPldapGetValues( mLDAP,
						currentMessage,
						(char *) kBaseKeySpaceAttr,
						&values );
				if( err != kPGPError_LDAPNoSuchAttribute )
				{
					if( IsPGPError( err ) )
						ThrowPGPError_( kPGPError_ServerOpenFailed );

					for( i = 0; IsntNull( values[i] ) && IsNull( mBaseKeySpaceDN ); i++ )
					{
						if( IsNull( mBaseKeySpaceDN ) )
						{
							mBaseKeySpaceDN = (char *) PGPNewData(
								PGPPeekContextMemoryMgr( mContext ),
								strlen( values[i] ) + 1,
								kPGPMemoryMgrFlags_Clear );
							if( IsNull( mBaseKeySpaceDN ) )
								ThrowPGPError_( kPGPError_ServerOpenFailed );

							pgpCopyMemory( values[i], mBaseKeySpaceDN,
								strlen( values[i] ) + 1 );
						}
					}

					(void) PGPFreeLDAPValues( values );
					values = NULL;
				}

				err = PGPldapNextEntry( mLDAP, currentMessage, &currentMessage );
				if( IsPGPError( err ) )
					ThrowPGPError_( kPGPError_ServerOpenFailed );
			}

			if( PGPldapMessageRefIsValid( firstMessage ) )
			{
				(void) PGPFreeLDAPMessage( firstMessage );
				firstMessage = kInvalidPGPldapMessageRef;
			}
		}

		/*
		 * If mBaseKeySpaceDN is still NULL, there wasn't a cn=PGPServerInfo
		 * entry.  The server hasn't been set up correctly, so we need to
		 * return an error.
		 */
		if( IsNull( mBaseKeySpaceDN ) )
			ThrowPGPError_( kPGPError_ServerOpenFailed );

		if( IsntNull( namingContexts ) )
			(void) PGPFreeLDAPValues( namingContexts );
	}
	catch( ... )
	{
		mIsOpen = false;
		if( PGPldapContextRefIsValid( mLDAP ) )
		{
			if( !mCanceled )
			{
				(void) PGPldapGetErrno( mLDAP, NULL, &szError, &result );
				SetErrorString( szError );
			}
		}
		
		if( IsntNull( serverInfoCN ) )
			(void) PGPFreeData( serverInfoCN );

		if( IsntNull( values ) )
			(void) PGPFreeLDAPValues( values );
		
		if( IsntNull( namingContexts ) )
			(void) PGPFreeLDAPValues( namingContexts );

		if( mCanceled )
		{
			mCanceled = false;
			ThrowPGPError_( kPGPError_UserAbort );
		}
		else
			throw;
	}

	return;
}


	void
CLDAPPGPKeyServer::Close( )
{
	StPreserveSocketsEventHandler	preserve(this);

	(void) pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Closing);
						
	if( PGPldapContextRefIsValid( mLDAP ) )
	{
		(void) PGPldapUnbind( mLDAP );

		(void) PGPFreeLDAPContext( mLDAP );
		mLDAP = kInvalidPGPldapContextRef;
	}

	if( IsntNull( mBaseKeySpaceDN ) )
		(void) PGPFreeData( mBaseKeySpaceDN );

	CKeyServer::Close();

	return;
}


	void
CLDAPPGPKeyServer::Query(
	PGPFilterRef				inFilterRef,
	PGPKeyDBRef *				outFoundKeys )
{
	StPreserveSocketsEventHandler		preserve(this);
	
	PGPError					err				= kPGPError_NoErr;
	char *						query			= NULL;
	const char *				pgpKeyAttrs[]	= { kPGPKeyAttr, NULL };
	PGPldapMessageRef			firstMessage	= kInvalidPGPldapMessageRef;
	PGPldapMessageRef			currentMessage	= kInvalidPGPldapMessageRef;
	PGPberValue **				berValues		= NULL;
	PGPBoolean					receivedBadKeys	= false;
	PGPBoolean					partialResults	= false;
	PGPKeyDBRef					singleKeyDB		= kInvalidPGPKeyDBRef;
	PGPKeyDBRef					foundKeys		= kInvalidPGPKeyDBRef;
	PGPKeySetRef				singleKeySet	= kInvalidPGPKeySetRef;
	char *						szError			= NULL;
	PGPldapResult				ldapResult		= kPGPldapResult_Success;
	PGPUInt32					i				= 0;

	SetErrorString( NULL );
	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		err = PGPLDAPQueryFromFilter( inFilterRef, &query );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Querying );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewLDAPMessage( mLDAP, &firstMessage );
		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		err = PGPldapSearchSync( mLDAP,
				mBaseKeySpaceDN,
				kPGPldapScope_Subtree,
				query,
				(char **) pgpKeyAttrs,
				FALSE,
				firstMessage );
		switch( err )
		{
			case kPGPError_NoErr:
				break;
			
			case kPGPError_LDAPInsufficientAccess:
				ThrowPGPError_( kPGPError_ServerAuthorizationFailed );
				break;
			
			case kPGPError_LDAPSizelimitExceeded:
			case kPGPError_LDAPTimelimitExceeded:
				partialResults = true;
				break;
			
			default:
				ThrowPGPError_( kPGPError_ServerSearchFailed );
				break;
		}

		err = pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_ProcessingResults );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewKeyDB( mContext, &foundKeys );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPldapFirstEntry( mLDAP, firstMessage, &currentMessage );
		if( IsPGPError( err ) )
			ThrowPGPError_( kPGPError_ServerSearchFailed );

		while( PGPldapMessageRefIsValid( currentMessage ) )
		{
			err = PGPldapGetValuesLen( mLDAP,
					currentMessage,
					(char *) kPGPKeyAttr,
					&berValues );
			if( err != kPGPError_LDAPNoSuchAttribute )
			{
				if( IsPGPError( err ) )
					ThrowPGPError_( kPGPError_ServerSearchFailed );

				for( i = 0; IsntNull( berValues[i] ); i++ )
				{
					err = PGPImport( mContext,
							&singleKeyDB,
							PGPOInputBuffer( mContext,
								berValues[i]->value,
								berValues[i]->length ),
							PGPOLastOption( mContext ) );
					if( IsPGPError( err ) )
						receivedBadKeys = true;
					else
					{
						err = PGPNewKeySet( singleKeyDB, &singleKeySet );
						ThrowIfPGPError_( err );

						err = PGPCopyKeys( singleKeySet, foundKeys, NULL );
						ThrowIfPGPError_( err );

						(void) PGPFreeKeySet( singleKeySet );
						singleKeySet = kInvalidPGPKeySetRef;

						(void) PGPFreeKeyDB( singleKeyDB );
						singleKeyDB = kInvalidPGPKeyDBRef;
					}
				}

				(void) PGPFreeLDAPValuesLen( berValues );
				berValues = NULL;
			}

			ThrowIfLDAPCanceled_();

			err = PGPldapNextEntry( mLDAP, currentMessage, &currentMessage );
			if( IsPGPError( err ) )
				ThrowPGPError_( kPGPError_ServerSearchFailed );
		}

		if( PGPldapMessageRefIsValid( firstMessage ) )
		{
			(void) PGPFreeLDAPMessage( firstMessage );
			firstMessage = kInvalidPGPldapMessageRef;
		}

		*outFoundKeys = foundKeys;


		if( PGPldapMessageRefIsValid( firstMessage ) )
		{
			(void) PGPFreeLDAPMessage( firstMessage );
			firstMessage = kInvalidPGPldapMessageRef;
		}

		if( IsntNull( berValues ) )
			(void) PGPFreeLDAPValuesLen( berValues );

		if( IsntNull( query ) )
			(void) PGPFreeData( query );
		
	}
	catch( ... )
	{
		if( PGPKeySetRefIsValid( singleKeySet ) )
			(void) PGPFreeKeySet( singleKeySet );

		if( PGPKeyDBRefIsValid( singleKeyDB ) )
			(void) PGPFreeKeyDB( singleKeyDB );

		if( PGPKeyDBRefIsValid( foundKeys ) )
			(void) PGPFreeKeyDB( foundKeys );

		if( PGPldapMessageRefIsValid( firstMessage ) )
			(void) PGPFreeLDAPMessage( firstMessage );

		if( IsntNull( berValues ) )
			(void) PGPFreeLDAPValuesLen( berValues );

		if( IsntNull( query ) )
			(void) PGPFreeData( query );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}

	if( receivedBadKeys )
		ThrowPGPError_( kPGPError_ServerBadKeysInSearchResults );

	if( partialResults )
		ThrowPGPError_( kPGPError_ServerPartialSearchResults );

	return;
}


	void
CLDAPPGPKeyServer::Upload(
	PGPKeyUploadPreference inSendPrivateKeys,
	PGPKeySetRef	inKeysToUpload,
	PGPKeySetRef *	outKeysThatFailed)
{
	StPreserveSocketsEventHandler		preserve(this);

	PGPKeyIterRef		keyIter				= kInvalidPGPKeyIterRef;
	PGPKeyListRef		keyList				= kInvalidPGPKeyListRef;
	PGPKeySetRef		errorKeySet			= kInvalidPGPKeySetRef;
	PGPKeyDBObjRef		key					= kInvalidPGPKeyDBObjRef;
	PGPKeyID			keyid;
	PGPKeyID *			keyidPtr			= NULL;
	char				szCertID[kPGPMaxKeyIDStringSize];
	char				szKeyID[kPGPMaxKeyIDStringSize];
	PGPKeyDBObjRef		userid				= kInvalidPGPKeyDBObjRef;
	char *				dn					= NULL;
	PGPldapMod **		mod					= NULL;
	PGPBoolean			exportPrivateKeys	= false;
	PGPBoolean			bDisabled			= false;
	char *				exportedKey			= NULL;
	PGPInt32			algorithm			= 0;
	char *				szAlgorithm			= NULL;
	PGPSize				numUserIDs			= 0;
	char **				szUserID			= NULL;
	PGPInt32			iType				= 0;
	PGPTime				creationPGPTime		= 0;
	time_t				creationTime		= 0;
	struct tm *			time				= NULL;
	char				szCreationTime[128];
	PGPTime				expirationPGPTime	= 0;
	time_t				expirationTime		= 0;
	char				szExpirationTime[128];
	PGPBoolean			bIsExportable		= false;
	PGPKeyDBObjRef		sig					= kInvalidPGPKeyDBObjRef;
	PGPSize				numSignatures		= 0;
	char **				szSignerID			= NULL;
	PGPBoolean			bRevoked			= false;
	PGPKeyDBObjRef		subkey				= kInvalidPGPKeyDBObjRef;
	PGPSize				numSubKeys			= 0;
	char **				szSubKeyID			= NULL;
	char **				szSubKeySize		= NULL;
	PGPSize				keySize				= 0;
	PGPSize				bufSize				= 0;
	PGPUInt32			i					= 0;
	PGPUInt32			j					= 0;
	PGPUInt32			attrNum				= 0;
	PGPMemoryMgrRef		mgr					= kInvalidPGPMemoryMgrRef;
	char *				szError				= NULL;
	PGPldapResult		ldapResult			= kPGPldapResult_Success;
	PGPError			err					= kPGPError_NoErr;


	SetErrorString( NULL );

	mgr = PGPPeekContextMemoryMgr( mContext );

	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	if( inSendPrivateKeys == kPGPPrivateKeyAllowed )
		exportPrivateKeys = true;

	try
	{
		err = pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Uploading );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewEmptyKeySet( PGPPeekKeySetKeyDB( inKeysToUpload ), &errorKeySet );
		ThrowIfPGPError_( err );

		/* Iterate through inKeysToUpload */
		err = PGPOrderKeySet( inKeysToUpload, kPGPKeyOrdering_Any, FALSE, &keyList );
		ThrowIfPGPError_( err );

		err = PGPNewKeyIter( keyList, &keyIter );
		ThrowIfPGPError_( err );

		while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Key, &key ) ) 
			== kPGPError_NoErr )
		{
			/* Upload the key */
			err = PGPExport( mContext,
					PGPOExportKeyDBObj( mContext, key ),
					PGPOExportPrivateKeys( mContext,
						exportPrivateKeys ),
					PGPOAllocatedOutputBuffer(
						mContext,
						(void **) &exportedKey,
						MAX_PGPSize,
						&bufSize ),
					PGPOExportFormat( mContext,
						mExportFormat ),
					PGPOLastOption( mContext ) );
			ThrowIfPGPError_( err );

			err = PGPGetKeyID( key, &keyid );
			ThrowIfPGPError_( err );

			err = PGPGetKeyIDString( &keyid, kPGPKeyIDString_Full, szCertID );
			ThrowIfPGPError_( err );

			err = PGPGetKeyIDString( &keyid, kPGPKeyIDString_Abbreviated, szKeyID );
			ThrowIfPGPError_( err );

			dn = (char *)PGPNewData( mgr, 
				strlen( kPGPCertIDAttr ) + sizeof( "=" ) + strlen( szCertID ) +
				sizeof( ", " ) +  strlen( mBaseKeySpaceDN ) + 1, 
				kPGPMemoryMgrFlags_Clear );
			if( IsNull( dn ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			sprintf( dn, "%s=%s, %s", kPGPCertIDAttr, szCertID + 2, mBaseKeySpaceDN );

			err = PGPGetKeyDBObjBooleanProperty( key, kPGPKeyProperty_IsDisabled, &bDisabled );
			ThrowIfPGPError_( err );

			err = PGPGetKeyDBObjNumericProperty( key, kPGPKeyProperty_AlgorithmID, &algorithm );
			ThrowIfPGPError_( err );

			switch( algorithm )
			{
				case kPGPPublicKeyAlgorithm_RSA:
				case kPGPPublicKeyAlgorithm_RSAEncryptOnly:
				case kPGPPublicKeyAlgorithm_RSASignOnly:
					szAlgorithm = (char *) kPGPAlgorithmString_RSA;
					break;
				case kPGPPublicKeyAlgorithm_ElGamal:
				case kPGPPublicKeyAlgorithm_DSA:
					szAlgorithm = (char *) kPGPAlgorithmString_DSSDH;
					break;
			}

			/* Count how many userids we have */
			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_UserID );
			ThrowIfPGPError_( err );

			numUserIDs = 0;
			while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_UserID, &userid ) )
				!= kPGPError_EndOfIteration )
			{
				err = PGPGetKeyDBObjNumericProperty( userid, 
						kPGPUserIDProperty_AttributeType, &iType );
				ThrowIfPGPError_( err );

				if( iType != kPGPAttribute_Image )
					numUserIDs++;
			}


			szUserID = (char **) PGPNewData( mgr, 
					( numUserIDs + 1 ) * sizeof( char * ), 
					kPGPMemoryMgrFlags_Clear );
			if( IsNull( szUserID ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			
			/* Now get the userids */
			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_UserID );
			ThrowIfPGPError_( err );

			for( i = 0; i < numUserIDs; )
			{
				err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_UserID, &userid );
				ThrowIfPGPError_( err );
				
				err = PGPGetKeyDBObjNumericProperty( userid, 
						kPGPUserIDProperty_AttributeType, &iType );
				ThrowIfPGPError_( err );

				if( iType != kPGPAttribute_Image )
				{
					err = PGPGetKeyDBObjAllocatedDataProperty( userid,
							kPGPUserIDProperty_Name,
							(void **) &szUserID[i++],
							&bufSize );
					ThrowIfPGPError_( err );
				}
			}
			szUserID[i] = NULL;

			
			err = PGPGetKeyDBObjTimeProperty( key,
					kPGPKeyProperty_Creation,
					&creationPGPTime );
			ThrowIfPGPError_( err );

			creationTime = PGPGetStdTimeFromPGPTime( creationPGPTime );

			time = gmtime( &creationTime );

			sprintf( szCreationTime, "%04i%02i%02i%02i%02i%02iZ",
				time->tm_year + 1900,
				time->tm_mon + 1,
				time->tm_mday,
				time->tm_hour,
				time->tm_min,
				time->tm_sec );


			err = PGPGetKeyDBObjTimeProperty( key,
					kPGPKeyProperty_Expiration,
					&expirationPGPTime );
			ThrowIfPGPError_( err );

			expirationTime = PGPGetStdTimeFromPGPTime( expirationPGPTime );

			time = gmtime( &expirationTime );

			sprintf( szExpirationTime, "%04i%02i%02i%02i%02i%02iZ",
				time->tm_year + 1900,
				time->tm_mon + 1,
				time->tm_mday,
				time->tm_hour,
				time->tm_min,
				time->tm_sec );


			
			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_UserID );
			ThrowIfPGPError_( err );

			numSignatures = 0;
			while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_UserID, &userid ) )
				!= kPGPError_EndOfIteration )
			{
				err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_Signature );
				ThrowIfPGPError_( err );

				while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Signature, &sig ) )
					!= kPGPError_EndOfIteration )
				{
					err = PGPGetKeyDBObjBooleanProperty( sig,
							kPGPSigProperty_IsExportable,
							&bIsExportable );
					ThrowIfPGPError_( err );
					
					if( bIsExportable )
						numSignatures++;
				}
			}


			szSignerID = (char **) PGPNewData( mgr,
					( numSignatures + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
			if( IsNull( szSignerID ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_UserID );
			ThrowIfPGPError_( err );

			i = 0;
			while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_UserID, &userid ) )
				!= kPGPError_EndOfIteration )
			{
				err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_Signature );
				ThrowIfPGPError_( err );

				while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Signature, &sig ) )
					!= kPGPError_EndOfIteration )
				{
					err = PGPGetKeyDBObjBooleanProperty( sig,
							kPGPSigProperty_IsExportable,
							&bIsExportable );
					ThrowIfPGPError_( err );
					
					if( bIsExportable )
					{
						err = PGPGetKeyDBObjAllocatedDataProperty( sig,
								kPGPSigProperty_KeyID,
								(void **) &keyidPtr,
								&bufSize );
						ThrowIfPGPError_( err );

						szSignerID[i] = (char *) PGPNewData( mgr,
								kPGPMaxKeyIDStringSize,
								kPGPMemoryMgrFlags_Clear );
						if( IsNull( szSignerID[i] ) )
							ThrowPGPError_( kPGPError_OutOfMemory );
						
						err = PGPGetKeyIDString( keyidPtr, kPGPKeyIDString_Full, szSignerID[i] );
						ThrowIfPGPError_( err );

						(void) PGPFreeData( keyidPtr );
						keyidPtr = NULL;

						/*
						 * Netscape Directory Server (and probably other LDAP servers)
						 * don't like having several identical attribute-value pairs,
						 * so I need to check szSignerID for duplicates.
						 */
						for( j = 0; j < i; j++ )
						{
							if( pgpCompareStringsIgnoreCase( szSignerID[j], szSignerID[i] ) == 0 )
								break;
						}
						if( j == i )	
						{
							/* We made it all the way through without finding a duplicate */
							i++;
						}
						else
						{
							(void) PGPFreeData( szSignerID[i] );
							szSignerID[i] = NULL;
						}
					}
				}
			}

			err = PGPGetKeyDBObjBooleanProperty( key, kPGPKeyProperty_IsRevoked, &bRevoked );
			ThrowIfPGPError_( err );

			
			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_SubKey );
			ThrowIfPGPError_( err );

			numSubKeys = 0;
			while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_SubKey, &subkey ) )
				!= kPGPError_EndOfIteration )
			{
				numSubKeys++;
			}

			szSubKeyID = (char **) PGPNewData( mgr,
					( numSubKeys + 1 ) * sizeof( char * ),
					kPGPMemoryMgrFlags_Clear );
			if( IsNull( szSubKeyID ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_SubKey );
			ThrowIfPGPError_( err );

			i = 0;
			while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_SubKey, &subkey ) )
				!= kPGPError_EndOfIteration )
			{
				err = PGPGetKeyDBObjAllocatedDataProperty( subkey,
						kPGPSubKeyProperty_KeyID,
						(void **) &keyidPtr,
						&bufSize );
				ThrowIfPGPError_( err );

				szSubKeyID[i] = (char *) PGPNewData( mgr,
						kPGPMaxKeyIDStringSize,
						kPGPMemoryMgrFlags_Clear );
				if( IsNull( szSubKeyID[i] ) )
					ThrowPGPError_( kPGPError_OutOfMemory );
				
				err = PGPGetKeyIDString( keyidPtr, kPGPKeyIDString_Full, szSubKeyID[i] );
				ThrowIfPGPError_( err );

				(void) PGPFreeData( keyidPtr );
				keyidPtr = NULL;

				/* Check for duplicates */
				for( j = 0; j < i; j++ )
				{
					if( pgpCompareStringsIgnoreCase( szSubKeyID[j], szSubKeyID[i] ) == 0 )
						break;
				}
				if( j == i )	
				{
					/* We made it all the way through without finding a duplicate */
					i++;
				}
				else
				{
					(void) PGPFreeData( szSubKeyID[i] );
					szSubKeyID[i] = NULL;
				}
			}


			if( numSubKeys > 0 )
			{
				szSubKeySize = (char **) PGPNewData( mgr,
						( numSubKeys + 1 ) * sizeof( char * ),
						kPGPMemoryMgrFlags_Clear );
				if( IsNull( szSubKeySize ) )
					ThrowPGPError_( kPGPError_OutOfMemory );

				err = PGPKeyIterRewind( keyIter, kPGPKeyDBObjType_SubKey );
				ThrowIfPGPError_( err );

				i = 0;
				while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_SubKey, &subkey ) )
					!= kPGPError_EndOfIteration )
				{
					err = PGPGetKeyDBObjNumericProperty( subkey,
							kPGPSubKeyProperty_Bits,
							(PGPInt32 *) &keySize );
					ThrowIfPGPError_( err );

					szSubKeySize[i] = (char *) PGPNewData( mgr,
							kMaxKeySizeStringLength,
							kPGPMemoryMgrFlags_Clear );
					if( IsNull( szSubKeySize[i] ) )
						ThrowPGPError_( kPGPError_OutOfMemory );
					
					sprintf( szSubKeySize[i], "%05i", (int) keySize );
					
					/* Check for duplicates */
					for( j = 0; j < i; j++ )
					{
						if( pgpCompareStringsIgnoreCase( szSubKeySize[j], szSubKeySize[i] ) == 0 )
							break;
					}
					if( j == i )	
					{
						/* We made it all the way through without finding a duplicate */
						i++;
					}
					else
					{
						(void) PGPFreeData( szSubKeySize[i] );
						szSubKeySize[i] = NULL;
					}
				}
			}
			else
			{
				/* Could be an RSA key or something that doesn't have subkeys */

				numSubKeys = 1;

				szSubKeySize = (char **) PGPNewData( mgr,
						2 * sizeof( char * ),
						kPGPMemoryMgrFlags_Clear );
				if( IsNull( szSubKeySize ) )
					ThrowPGPError_( kPGPError_OutOfMemory );

				err = PGPGetKeyDBObjNumericProperty( key,
						kPGPKeyProperty_Bits,
						(PGPInt32 *) &keySize );
				ThrowIfPGPError_( err );

				szSubKeySize[0] = (char *) PGPNewData( mgr,
						kMaxKeySizeStringLength,
						kPGPMemoryMgrFlags_Clear );
				if( IsNull( szSubKeySize[i] ) )
					ThrowPGPError_( kPGPError_OutOfMemory );
					
				sprintf( szSubKeySize[0], "%05i", (int) keySize );

				szSubKeySize[1] = NULL;
			}



			mod = (PGPldapMod **) PGPNewData( mgr,
					sizeof( PGPldapMod * ) * ( kPGPNumberOfAttrs + 1 ),
					kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			for( i = 0; i < kPGPNumberOfAttrs; i++ )
			{
				mod[i] = (PGPldapMod *) PGPNewData( mgr,
							sizeof( PGPldapMod ),
							kPGPMemoryMgrFlags_Clear );
				if( IsNull( mod[i] ) )
					ThrowPGPError_( kPGPError_OutOfMemory );
			}

			attrNum = 0;

			/* objectclass */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kObjectclassAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = (char *) kPGPKeyObjectClass;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpcertid */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPCertIDAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = szCertID + 2;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpkey */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeyAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = exportedKey;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpdisabled */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPDisabledAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = (char *) ( bDisabled ? "1" : "0" );
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpkeyid */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeyIDAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = szKeyID + 2;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpkeytype */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeyTypeAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = szAlgorithm;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpuserid */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPUserIDAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									( numUserIDs + 1 ) * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			for( i = 0; IsntNull( szUserID[i] ); i++ )
				mod[attrNum]->value[i] = szUserID[i];
			mod[attrNum]->value[i] = NULL;
			attrNum++;

			/* pgpkeycreatetime */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeyCreateTime;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = szCreationTime;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpkeycreatetime */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeyExpireTime;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = szExpirationTime;
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpsignerid */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPSignerIDAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									( numSignatures + 1 ) * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			for( i = 0; IsntNull( szSignerID[i] ); i++ )
				mod[attrNum]->value[i] = szSignerID[i] + 2;
			mod[attrNum]->value[i] = NULL;
			attrNum++;

			/* pgprevoked */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPRevokedAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[attrNum]->value[0] = (char *) ( bRevoked ? "1" : "0" );
			mod[attrNum]->value[1] = NULL;
			attrNum++;

			/* pgpsubkeyid */
			if( IsntNull( szSubKeyID[0] ) )
			{
				mod[attrNum]->op	= kPGPldapModOp_Add;
				mod[attrNum]->type	= (char *) kPGPSubKeyIDAttr;
				mod[attrNum]->value	= (char **) PGPNewData( mgr,
										( numSubKeys + 1 ) * sizeof( char * ),
										kPGPMemoryMgrFlags_Clear );
				if( IsNull( mod[attrNum]->value ) )
					ThrowPGPError_( kPGPError_OutOfMemory );
				for( i = 0; IsntNull( szSubKeyID[i] ); i++ )
					mod[attrNum]->value[i] = szSubKeyID[i] + 2;
				mod[attrNum]->value[i] = NULL;
				attrNum++;
			}

			/* pgpkeysize */
			mod[attrNum]->op	= kPGPldapModOp_Add;
			mod[attrNum]->type	= (char *) kPGPKeySizeAttr;
			mod[attrNum]->value	= (char **) PGPNewData( mgr,
									( numSubKeys + 1 ) * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[attrNum]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			for( i = 0; IsntNull( szSubKeySize[i] ); i++ )
				mod[attrNum]->value[i] = szSubKeySize[i];
			mod[attrNum]->value[i] = NULL;
			attrNum++;

			/* NULL */
			while( attrNum < kPGPNumberOfAttrs )
			{
				if( IsntNull( mod[attrNum] ) )
					(void) PGPFreeData( mod[attrNum] );
				mod[attrNum++] = NULL;
			}

			/*
			 * If the DN already exists, this will have no effect.  New
			 * attributes or values will NOT be added.  In order to do this,
			 * we need to PGPldapModify the item on the server.
			 */
			err = PGPldapAddSync( mLDAP, dn, mod );
			if( err == kPGPError_LDAPAlreadyExists )
			{
				/*
				 * It's already there.  We need to replace what's already there.
				 * Note that this will add new attributes and new values to
				 * existing attributes.
				 */
				for( i = 0; IsntNull( mod[i] ); i++ )
					mod[i]->op = kPGPldapModOp_Replace;

				err = PGPldapModifySync( mLDAP, dn,	mod );
			}

			/* If upload fails, put key in errorKeySet */
			if( IsPGPError( err ) )
			{
				(void) PGPAddKey( key, errorKeySet );
				ThrowPGPError_( err );
			}

			if( IsntNull( dn ) )
			{
				(void) PGPFreeData( dn );
				dn = NULL;
			}

			if( IsntNull( exportedKey ) )
			{
				(void) PGPFreeData( exportedKey );
				exportedKey = NULL;
			}

			if( IsntNull( mod ) )
			{
				for( i = 0; IsntNull( mod[i] ); i++ )
				{
					if( IsntNull( mod[i]->value ) )
						(void) PGPFreeData( mod[i]->value );
					if( IsntNull( mod[i] ) )
						(void) PGPFreeData( mod[i] );
				}
				(void) PGPFreeData( mod );
				mod = NULL;
			}

			if( IsntNull( szUserID ) )
			{
				for( i = 0; IsntNull( szUserID[i] ); i++ )
					(void) PGPFreeData( szUserID[i] );
				(void) PGPFreeData( szUserID );
				szUserID = NULL;
			}

			if( IsntNull( szSignerID ) )
			{
				for( i = 0; IsntNull( szSignerID[i] ); i++ )
					(void) PGPFreeData( szSignerID[i] );
				(void) PGPFreeData( szSignerID );
				szSignerID = NULL;
			}

			if( IsntNull( szSubKeyID ) )
			{
				for( i = 0; IsntNull( szSubKeyID[i] ); i++ )
					(void) PGPFreeData( szSubKeyID[i] );
				(void) PGPFreeData( szSubKeyID );
				szSubKeyID = NULL;
			}

			if( IsntNull( szSubKeySize ) )
			{
				for( i = 0; IsntNull( szSubKeySize[i] ); i++ )
					(void) PGPFreeData( szSubKeySize[i] );
				(void) PGPFreeData( szSubKeySize );
				szSubKeySize = NULL;
			}

		}

		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );
		keyIter = kInvalidPGPKeyIterRef;

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );
		keyList = kInvalidPGPKeyListRef;

		if( IsntNull( outKeysThatFailed ) )
			*outKeysThatFailed = errorKeySet;
	}
	catch( ... )
	{
		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );

		if( IsntNull( dn ) )
			(void) PGPFreeData( dn );

		if( IsntNull( exportedKey ) )
			(void) PGPFreeData( exportedKey );

		if( IsntNull( mod ) )
		{
			for( i = 0; IsntNull( mod[i] ); i++ )
			{
				if( IsntNull( mod[i]->value ) )
					(void) PGPFreeData( mod[i]->value );
				if( IsntNull( mod[i] ) )
					(void) PGPFreeData( mod[i] );
			}
			(void) PGPFreeData( mod );
		}

		if( IsntNull( szUserID ) )
		{
			for( i = 0; IsntNull( szUserID[i] ); i++ )
				(void) PGPFreeData( szUserID[i] );
			(void) PGPFreeData( szUserID );
		}

		if( IsntNull( szSignerID ) )
		{
			for( i = 0; IsntNull( szSignerID[i] ); i++ )
				(void) PGPFreeData( szSignerID[i] );
			(void) PGPFreeData( szSignerID );
		}

		if( IsntNull( szSubKeyID ) )
		{
			for( i = 0; IsntNull( szSubKeyID[i] ); i++ )
				(void) PGPFreeData( szSubKeyID[i] );
			(void) PGPFreeData( szSubKeyID );
		}

		if( IsntNull( szSubKeySize ) )
		{
			for( i = 0; IsntNull( szSubKeySize[i] ); i++ )
				(void) PGPFreeData( szSubKeySize[i] );
			(void) PGPFreeData( szSubKeySize );
		}

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}

	return;
}


	void
CLDAPPGPKeyServer::Delete(
	PGPKeySetRef	inKeysToDelete,
	PGPKeySetRef *	outKeysThatFailed)
{
	StPreserveSocketsEventHandler		preserve(this);

	PGPKeyIterRef		keyIter				= kInvalidPGPKeyIterRef;
	PGPKeyListRef		keyList				= kInvalidPGPKeyListRef;
	PGPKeySetRef		errorKeySet			= kInvalidPGPKeySetRef;
	PGPKeyDBObjRef		key					= kInvalidPGPKeyDBObjRef;
	PGPKeyID			keyid;
	char				szCertID[kPGPMaxKeyIDStringSize];
	char *				dn					= NULL;
	char *				szError				= NULL;
	PGPldapResult		ldapResult			= kPGPldapResult_Success;
	PGPError			err					= kPGPError_NoErr;


	SetErrorString( NULL );

	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		err = pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Deleting );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewEmptyKeySet( PGPPeekKeySetKeyDB( inKeysToDelete ), &errorKeySet );
		ThrowIfPGPError_( err );

		/* Iterate through inKeysToDelete */
		err = PGPOrderKeySet( inKeysToDelete, kPGPKeyOrdering_Any, FALSE, &keyList );
		ThrowIfPGPError_( err );

		err = PGPNewKeyIter( keyList, &keyIter );
		ThrowIfPGPError_( err );

		while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Key, &key ) ) 
			== kPGPError_NoErr )
		{
			/* Delete the key */
			err = PGPGetKeyID( key, &keyid );
			ThrowIfPGPError_( err );

			err = PGPGetKeyIDString( &keyid, kPGPKeyIDString_Full, szCertID );
			ThrowIfPGPError_( err );

			dn = (char *)PGPNewData( PGPPeekContextMemoryMgr( mContext ), 
				strlen( kPGPCertIDAttr ) + sizeof( "=" ) + strlen( szCertID ) +
				sizeof( ", " ) +  strlen( mBaseKeySpaceDN ) + 1, 
				kPGPMemoryMgrFlags_Clear );
			if( IsNull( dn ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			sprintf( dn, "%s=%s, %s", kPGPCertIDAttr, szCertID + 2, mBaseKeySpaceDN );

			err = PGPldapDeleteSync( mLDAP, dn );

			/* If delete fails, put key in errorKeySet */
			if( IsPGPError( err ) )
			{
				err = PGPAddKey( key, errorKeySet );
				ThrowIfPGPError_( err );
			}

			if( IsntNull( dn ) )
			{
				(void) PGPFreeData( dn );
				dn = NULL;
			}

		}

		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );
		keyIter = kInvalidPGPKeyIterRef;

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );
		keyList = kInvalidPGPKeyListRef;

		if( IsntNull( outKeysThatFailed ) )
			*outKeysThatFailed = errorKeySet;
	}
	catch( ... )
	{
		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );

		if( IsntNull( dn ) )
			(void) PGPFreeData( dn );

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}

	return;
}



	void
CLDAPPGPKeyServer::Disable(
	PGPKeySetRef	inKeysToDisable,
	PGPKeySetRef *	outKeysThatFailed)
{
	StPreserveSocketsEventHandler		preserve(this);

	PGPKeyIterRef		keyIter				= kInvalidPGPKeyIterRef;
	PGPKeyListRef		keyList				= kInvalidPGPKeyListRef;
	PGPKeySetRef		errorKeySet			= kInvalidPGPKeySetRef;
	PGPKeyDBObjRef		key					= kInvalidPGPKeyDBObjRef;
	PGPKeyID			keyid;
	char				szCertID[kPGPMaxKeyIDStringSize];
	char *				dn					= NULL;
	char *				szError				= NULL;
	PGPldapMod			disableMod;
	PGPldapMod *		mod[]				= { &disableMod, NULL };
	PGPldapResult		ldapResult			= kPGPldapResult_Success;
	PGPError			err					= kPGPError_NoErr;


	SetErrorString( NULL );

	if( !mIsOpen )
		ThrowPGPError_( kPGPError_ServerNotOpen );

	try
	{
		err = pgpEventKeyServer( mContext,
				mEventHandler,
				mEventHandlerData,
				(PGPKeyServerRef) this,
				kPGPKeyServerState_Deleting );
		ThrowIfPGPErrorOrLDAPCanceled_( err );

		err = PGPNewEmptyKeySet( PGPPeekKeySetKeyDB( inKeysToDisable ), &errorKeySet );
		ThrowIfPGPError_( err );

		/* Iterate through inKeysToDisable */
		err = PGPOrderKeySet( inKeysToDisable, kPGPKeyOrdering_Any, FALSE, &keyList );
		ThrowIfPGPError_( err );

		err = PGPNewKeyIter( keyList, &keyIter );
		ThrowIfPGPError_( err );

		while( ( err = PGPKeyIterNextKeyDBObj( keyIter, kPGPKeyDBObjType_Key, &key ) ) 
			== kPGPError_NoErr )
		{
			/* Disable the key */
			err = PGPGetKeyID( key, &keyid );
			ThrowIfPGPError_( err );

			err = PGPGetKeyIDString( &keyid, kPGPKeyIDString_Full, szCertID );
			ThrowIfPGPError_( err );

			dn = (char *)PGPNewData( PGPPeekContextMemoryMgr( mContext ), 
				strlen( kPGPCertIDAttr ) + sizeof( "=" ) + strlen( szCertID ) +
				sizeof( ", " ) +  strlen( mBaseKeySpaceDN ) + 1, 
				kPGPMemoryMgrFlags_Clear );
			if( IsNull( dn ) )
				ThrowPGPError_( kPGPError_OutOfMemory );

			sprintf( dn, "%s=%s, %s", kPGPCertIDAttr, szCertID + 2, mBaseKeySpaceDN );

			mod[0]->op		= kPGPldapModOp_Replace;
			mod[0]->type	= (char *) kPGPDisabledAttr;
			mod[0]->value	= (char **) PGPNewData( PGPPeekContextMemoryMgr( mContext ),
									2 * sizeof( char * ),
									kPGPMemoryMgrFlags_Clear );
			if( IsNull( mod[0]->value ) )
				ThrowPGPError_( kPGPError_OutOfMemory );
			mod[0]->value[0] = (char *) "1";
			mod[0]->value[1] = NULL;

			mod[1] = NULL;

			err = PGPldapModifySync( mLDAP, dn, mod );

			/* If disable fails, put key in errorKeySet */
			if( IsPGPError( err ) )
			{
				err = PGPAddKey( key, errorKeySet );
				ThrowIfPGPError_( err );
			}

			if( IsntNull( dn ) )
			{
				(void) PGPFreeData( dn );
				dn = NULL;
			}

			if( IsntNull( mod[0]->value ) )
				(void) PGPFreeData( mod[0]->value );
			mod[0]->value = NULL;

		}

		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );
		keyIter = kInvalidPGPKeyIterRef;

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );
		keyList = kInvalidPGPKeyListRef;

		if( IsntNull( outKeysThatFailed ) )
			*outKeysThatFailed = errorKeySet;
	}
	catch( ... )
	{
		if( PGPKeyIterRefIsValid( keyIter ) )
			(void) PGPFreeKeyIter( keyIter );

		if( PGPKeyListRefIsValid( keyList ) )
			(void) PGPFreeKeyList( keyList );

		if( IsntNull( dn ) )
			(void) PGPFreeData( dn );

		if( IsntNull( mod[0]->value ) )
			(void) PGPFreeData( mod[0]->value );
		mod[0]->value = NULL;

		if( mCanceled )
			ThrowPGPError_( kPGPError_UserAbort );
		else 
		{
			(void) PGPldapGetErrno( mLDAP, NULL, &szError, &ldapResult );
			SetErrorString( szError );
			throw;
		}
	}

	return;
}





/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/
